#include <stdlib.h>
#include "syndrome.h"
#include <getopt.h>
#include <err.h>
#include <string.h>
#include "trellis.h"
#include "probe.h"

#define PROGNAME "solidCaller"

static struct option longopts[] = {
        {"fasta", no_argument, NULL, 'a'},
        {"color", no_argument, NULL, 'c'},
	{"generator", required_argument, NULL, 'g'},
        {"noecc", no_argument, NULL, 'n'},
	{"initial", required_argument, NULL, 'i'},
	{"method", required_argument, NULL, 'm'},
	{"syndrome", no_argument, NULL, 's'},
	{"help", no_argument, NULL, 'h'},
	{"licence", no_argument, NULL, 0},
	{"license", no_argument, NULL, 0},
	{NULL,0,NULL,0}
};

enum calling_methods { METHOD_COLOR, METHOD_INVERT, METHOD_MAP, METHOD_ML };
char * calling_method_strings[] = { "color","invert","map","ml"};

typedef struct options {
	base initBase;
	bool showSyndrome,useEcc,outputBase,outputFastq;
	base probeGen[MAXCODE];
	enum calling_methods method;
} OPT;

OPT opt = { baseT, false, true, true, true, {baseC,baseT,baseA,baseT}, METHOD_MAP };

void fprint_usage(FILE * fp){
	fputs(
"\t" PROGNAME "\n"
"Call bases from SOLiD4 color data\n"
"\n"
"Usage:\n"
"\t" PROGNAME " -acns [-g probeGenerator] [-i initBase] [-m method] file\n"
"\t" PROGNAME " --help\n"
"\t" PROGNAME " --licence\n"
"\n"
PROGNAME " reads from a tim formated file containing colors and qualities\n"
"and prints a fastq formated base translation to stdout.\n"
"\n"
"Example: " PROGNAME " -m invert solid.tim\n"
	,fp);
}

void fprint_help(FILE * fp){
	fputs(
"\n"
"-a, --fasta [default:false]\n"
"\tOutput fasta even if qualities available.\n"
"\n"
"-c, --color [default: false]\n"
"\tDecode into colorspace\n"
"\n"
"-g, --generator [default: CTAT]\n"
"\tGenerator of the code for the second probe set.\n"
"\n"
"-i, --initial initialBase [default: T]\n"
"\tInitial base for direct translation of color-space information\n"
"Also used for calculating the first element of syndrome.\n"
"\n"
"-m, --method method [default: invert]\n"
"\tMethod used to convert color-space information into base-space.\n"
"Options are: color invert trellis\n"
"  color   Direct translation of color-space into base-space, ignoring\n"
"information in second probe set.\n"
"  invert  Invert the color-space + ecc-space sequence on a block-wise\n"
"basis.\n"
"  map Calling using dynamic programming on a trellis, using\n"
"color-space information from both probe sets.\n"
"  ml Calling most likely base sequence from trellis, using\n"
"color-space information from both probe sets.\n"
"\n"
"-n, --noecc [default: false]\n"
"\tDon't use ECC information to make predictions\n"
"\n"
"-s, --syndrome [default: false]\n"
"\tShow syndromes.\n"
"\n"
"-h, --help\n"
"\tDisplay information about usage and exit.\n"
"\n"
"--licence, --license\n"
"\tDisplay licence terms and exit.\n"
, fp);
}


void fprint_licence (FILE * fp){
	fputs(
		"  " PROGNAME " : Call bases from SOLiD4 color data\n"
		#include "copyright.inc"
	,fp);
}



void parse_options(int argc, char * argv[]){
	int ch,m;
	while ((ch = getopt_long(argc, argv, "acg:i:m:nsh", longopts, NULL)) != -1){
		switch(ch){
		        case 'a': opt.outputFastq = false;
				  break;
			case 'c': opt.outputBase = false;
				  break;
			case 'g': if(strlen(optarg)>MAXCODE){
					  warnx("Generator for code is too long, will truncate to %d positions",MAXCODE);
				  } else if (strlen(optarg)<MAXCODE){
					  warnx("Generator for code is too short, will pad with zeros");
				  }
				  int i = 0;
				  for ( i=0 ; i<strlen(optarg) ; i++){
					  base b = char2base(optarg[i]);
					  if(b==baseN){
						  errx(EXIT_FAILURE,"Generator %s contains illegal base %c",optarg,optarg[i]);
					  }
					  opt.probeGen[i] = b;
				  }
				  for ( ; i<MAXCODE ; i++){
					  opt.probeGen[i] = baseA;
				  }
				  break;
			case 'i': opt.initBase = char2base(optarg[0]);
				  break;
			case 'm':
				  m = (0==strcmp(optarg,calling_method_strings[0]))?1:0;
				  m += (0==strcmp(optarg,calling_method_strings[1]))?2:0;
				  m += (0==strcmp(optarg,calling_method_strings[2]))?4:0;
				  m += (0==strcmp(optarg,calling_method_strings[3]))?8:0;
				  switch(m){
					  case 0: errx(EXIT_FAILURE,"Unrecognised method string %s",optarg);
					  case 1: opt.method = METHOD_COLOR; break;
					  case 2: opt.method = METHOD_INVERT; break;
					  case 4: opt.method = METHOD_MAP; break;
					  case 8: opt.method = METHOD_ML; break;
					  default: errx(EXIT_FAILURE,"Ambiguous method string %s",optarg);
				  }
				  break;
			case 'n': opt.useEcc = false;
				  break;
			case 's': opt.showSyndrome = true;
				  break;
			case 'h': fprint_usage(stdout);
				  fprint_help(stdout);
				  exit(EXIT_SUCCESS);
			case 0:	  fprint_licence(stdout);
				  exit(EXIT_SUCCESS);
			default:  warnx("Unrecognised option %s",argv[optind]);
				  fprint_usage(stderr);
				  exit(EXIT_FAILURE);
		}
	}
}


int main ( int argc, char * argv[] ){
	parse_options(argc,argv);
	argc -= optind;
	argv += optind;
	init_probes(opt.probeGen);

	FILE * fp = stdin;
	size_t ctr=0;
	do {
		// Open next file, or use stdin
		if(argc>0){
			fp = fopen(argv[0],"r");
			if(NULL==fp){
				warnx("Failed to open %s for input",argv[0]);
			} else {
				fprintf(stderr,"Reading from %s\n",argv[0]);
			}
		} else {
			fputs("Reading from stdin\n",stderr);
		}

		/* Read file and convert sequence */
		SEQ seq = NULL, bseq=NULL;
		base * baseseq = NULL;
		while( (seq = read_seq(fp))!=NULL ){
	                double * probs = construct_trellis_probs(seq,opt.useEcc);
			switch(opt.method){
				case METHOD_COLOR: baseseq = call_sequence_from_colors(seq,opt.initBase,opt.outputBase,baseseq); break;
				case METHOD_INVERT: baseseq = call_sequence(seq,opt.initBase,opt.outputBase,baseseq); break;
				case METHOD_MAP: bseq = map_decoder(seq,opt.initBase,opt.outputBase,probs); break;
				case METHOD_ML: bseq = viterbi_decoder(seq,opt.initBase,opt.outputBase,probs); break;
			}
			//baseseq = call_sequence_from_colors(bseq,opt.initBase,true,baseseq);
			fprint_SEQ(stdout,bseq,opt.outputFastq&&has_qualities(bseq),true);
			/*printf(">%s\n",seq->name);
			fprint_baseVec(stdout,baseseq,seq->len);
			fputc('\n',stdout);*/
			/*printf(">%s\n",seq->name);
			// MAP
			bseq = map_decoder(seq,opt.initBase,true,probs);
			fprint_baseVec(stdout,bseq->base,seq->len);
			printf("\t%6.4f\tMAP\n",sequence_likelihood(bseq,opt.initBase,probs));
			free_SEQ(bseq); bseq=NULL;
			// MAP-color
			bseq = map_decoder(seq,opt.initBase,false,probs);
			baseseq = call_sequence_from_colors(bseq,opt.initBase,true,baseseq);
			for(int i=0 ; i<seq->len; i++){ bseq->base[i] = baseseq[i];}
			fprint_baseVec(stdout,bseq->base,seq->len);
			printf("\t%6.4f\tMAPc\n",sequence_likelihood(bseq,opt.initBase,probs));
			free_SEQ(bseq); bseq=NULL;
			bseq = viterbi_decoder(seq,opt.initBase,true,probs);
			fprint_baseVec(stdout,bseq->base,seq->len);
			printf("\t%6.4f\tML\n",sequence_likelihood(bseq,opt.initBase,probs));
			free_SEQ(bseq); bseq=NULL;
			bseq = viterbi_decoder(seq,opt.initBase,false,probs);
			baseseq = call_sequence_from_colors(bseq,opt.initBase,true,baseseq);
			for(int i=0 ; i<seq->len; i++){ bseq->base[i] = baseseq[i];}
			fprint_baseVec(stdout,bseq->base,seq->len);
			printf("\t%6.4f\tMLc\n",sequence_likelihood(bseq,opt.initBase,probs));*/



			free_SEQ(bseq); bseq=NULL;


			free_SEQ(seq); seq = NULL;
			free(probs);
			ctr++;
			if((ctr%10000)==0){
			   fprintf(stderr,"\rDone %8u",ctr);
			}
		}
		free(baseseq);
		fputc('\n',stderr);

		// Close file and set up for next
		fclose(fp);
		argc--;
		argv++;
	} while(argc>0);
}

