#include <stdlib.h>
#include <string.h>
#include "sequence.h"
#include "gf4.h"

/** Call sequence of a block using color and ecc information
 *  Works by assuming that colors are correct and inverting,
 *  ignoring the parity color.
 */
base * call_block( const base * cseq, const base ecc, base * seq){
	base bseq[BLOCK_SIZE] = {0};
	if(NULL==cseq){ return NULL; }

	// Create block color-space sequence, ignoring 5th parity color
	for ( int i=0 ; i<BLOCK_SIZE ; i++){ bseq[i] = cseq[i];}
	bseq[BLOCK_SIZE-1] = ecc;
	// Convert to base-space sequence using inverse marix
	return gf4_vector_matrix(bseq,decodeMat,BLOCK_SIZE,BLOCK_SIZE,seq);
}

/** Call sequence directly from normal colour stream, starting at initial base initB
 */
base * call_sequence_from_colors(SEQ seq, const base initB, const bool basespace, base * bseq){
	if(NULL==seq){ return NULL;}
	if(NULL==bseq){
		bseq = calloc(seq->len,sizeof(base));
		if(NULL==bseq){ return NULL; }
	}
	if(basespace){
	    // Initialise
	    bseq[0] = gf4_add(initB,seq->base[0]);
	    // Call base. c = b1 + b2 therefore b2 = c + b1
	    for(int i=1 ; i<seq->len ; i++){
		bseq[i] = gf4_add(bseq[i-1],seq->base[i]);
	    }
	} else {
	    for(int i=0 ; i<seq->len ; i++){
	        bseq[i] = seq->base[i];
	    }
	}
	return bseq;
}

base * call_sequence(SEQ seq, const base initB, const bool basespace, base * bseq){
	if(NULL==seq){ return NULL; }
	if(!has_ecc(seq)){ return call_sequence_from_colors(seq,initB,basespace,bseq); }

	if(NULL==bseq){
		bseq = calloc(seq->len,sizeof(base));
		if(NULL==bseq){ return NULL; }
	}

	if(basespace){
	    for ( int pos=0 ; pos<seq->len ; pos+=BLOCK_SIZE){
	        call_block(seq->base+1+pos,seq->ebase[2+pos],bseq+pos);
	    }
	} else {
	    for ( int pos=0 ; pos<seq->len ; pos++){
	        bseq[pos] = seq->base[pos];
	    }
	}
	return bseq;
}

base * get_syndromes(SEQ seq, const base * bseq, const base initB, base * synseq){
	if(NULL==seq){ return NULL;}
	if(NULL==bseq){ return NULL;}
	unsigned int nsyndrome = seq->len/BLOCK_SIZE;
	if(NULL==synseq){
		synseq = calloc(nsyndrome,sizeof(base));
		if(NULL==synseq){ return NULL; }
	}

	// First syndrome straddles initial base and first block
	synseq[0] = gf4_add(gf4_add(initB,bseq[0]),seq->base[0]);
	for ( int i=1 ; i<nsyndrome ; i++){
		const unsigned int pos = i*BLOCK_SIZE;
		synseq[i] = gf4_add(gf4_add(bseq[pos-1],bseq[pos]),seq->base[pos]);
	}
	return synseq;
}

#ifdef TEST
char name[] = "427_19_1309_F3";
unsigned int len = 50;
char fastq_base[] = {
	'0','0','2','1','0','0','3','1','3','1',
	'2','0','0','1','3','0','3','1','3','0',
	'3','1','2','1','3','0','3','3','3','3',
	'0','3','1','2','2','1','1','3','2','2',
//	'0','3','2','2','2','1','1','3','2','2',
	'1','0','1','0','2','0','0','0','0','2'};

char fastq_ecc[] = {
	'.','.','3','.','.','.','.','1','.','.',
	'.','.','0','.','.','.','.','2','.','.',
	'.','.','1','.','.','.','.','1','.','.',
//	'.','.','1','.','.','.','.','3','.','.',
	'.','.','3','.','.','.','.','1','.','.',
	'.','.','3','.','.','.','.','1','.','.'};
//	'.','.','2','.','.','.','.','1','.','.'};

phredq pqual[] = {
	22,30,19, 9,21, 27,28,16,25,21,
	24,25,20,20,24, 26,26,23,11,25,
	23,16,18, 4,19, 28,20,24, 7,28,
	27,22,25, 4,20, 12, 9,29, 4,24,
	20,19,20, 4,25, 21,25,25, 4,23};
phredq equal[] = {
	-1,-1,22,-1,-1,-1,-1,26,-1,-1,
	-1,-1,28,-1,-1,-1,-1,26,-1,-1,
	-1,-1,22,-1,-1,-1,-1,22,-1,-1,
	-1,-1,22,-1,-1,-1,-1,4,-1,-1,
	-1,-1,22,-1,-1,-1,-1,24,-1,-1};

int main ( int argc, char * argv[] ){
	SEQ seq = new_SEQ(len,true,true);

	seq->name = calloc(1+strlen(name),sizeof(char));
	strcpy(seq->name,name);
	for ( int i=0 ; i<len ; i++){
		seq->base[i] = char2base(fastq_base[i]);
		seq->qual[i] = phredq2qual(pqual[i]);
	}
	for ( int i=0 ; i<len ; i++){
		// Add T since values in data are probe space, not template space
		seq->ebase[i] = gf4_add(baseT,char2base(fastq_ecc[i]));
		seq->equal[i] = phredq2qual(equal[i]);
	}
	fprint_SEQ(stdout,seq,true,false);

	fputs("\nTranslations of colour-space sequence\n",stdout);
	base * baseseq = call_sequence_from_colors(seq,baseA,NULL);
	fputs("A:",stdout);
	fprint_baseVec(stdout,baseseq,seq->len);
	fputc('\n',stdout);
	baseseq = call_sequence_from_colors(seq,baseC,baseseq);
	fputs("C:",stdout);
	fprint_baseVec(stdout,baseseq,seq->len);
	fputc('\n',stdout);
	baseseq = call_sequence_from_colors(seq,baseG,baseseq);
	fputs("G:",stdout);
	fprint_baseVec(stdout,baseseq,seq->len);
	fputc('\n',stdout);
	baseseq = call_sequence_from_colors(seq,baseT,baseseq);
	fputs("T:",stdout);
	fprint_baseVec(stdout,baseseq,seq->len);
	fputc('\n',stdout);


	fputs("\nTranslation of sequence by inversion\n  ",stdout);
	call_sequence(seq,baseT,baseseq);
	fprint_baseVec(stdout,baseseq,seq->len);
	fputc('\n',stdout);
	fputs("Syndromes\n",stdout);
	base * synseq = get_syndromes(seq,baseseq,baseT,NULL);
	fprint_baseVec(stdout,synseq,seq->len/BLOCK_SIZE);
	fputc('\n',stdout);
}
#endif /* TEST */

