#include <stdlib.h>
#include <string.h>
#include "sequence.h"
#include "err.h"
#include "gf4.h"

SEQ new_SEQ(const unsigned int len, const bool hasQual, const bool hasECC){
	SEQ seq = calloc(1,sizeof(*seq));
	if(NULL==seq){ return NULL; }

	seq->len = len;

	seq->base = calloc(len,sizeof(base)); // Initialised to 0=baseA
	if(NULL==seq->base){ goto cleanup; }

	if(hasQual){
		seq->qual = calloc(len,sizeof(qual));
		if(NULL==seq->qual){
			goto cleanup;
		}
	}

	if(hasECC){
		seq->ebase = calloc(len,sizeof(base));
		if(NULL==seq->ebase){ goto cleanup; }
		if(hasQual){
			seq->equal = calloc(len,sizeof(qual));
			if(NULL==seq->equal){ goto cleanup;}
		}
	}

	return seq;
cleanup:
	free_SEQ(seq);
	return NULL;
}

void free_SEQ(SEQ seq){
	if(NULL==seq){ return;}
	free(seq->name);
	free(seq->base);
	free(seq->ebase);
	free(seq->qual);
	free(seq->equal);
	free(seq);
}

void fprint_SEQ(FILE * fp,SEQ seq, const bool showQual, const bool baseSpace){
	if(NULL==fp){ return; }
	if(NULL==seq){ return; }

	fputc(showQual?'@':'>',fp);
	fputs((NULL!=seq->name)?seq->name:"Unnamed",fp);
	fputc('\n',fp);
	if(baseSpace){
		for(int i=0 ; i<seq->len ; i++){
			fprint_base(fp,seq->base[i]);
		}
		if(has_ecc(seq)){
			fputc('\n',fp);
			for(int i=0 ; i<seq->len ; i++){
				fprint_base(fp,seq->ebase[i]);
			}
		}
	} else {
		for(int i=0 ; i<seq->len ; i++){
			fprint_color(fp,seq->base[i]);
		}
		if(has_ecc(seq)){
			fputc('\n',fp);
			for(int i=0 ; i<seq->len ; i++){
				fprint_color(fp,seq->ebase[i]);
			}
		}
	}
	fputc('\n',fp);

	if(showQual){
		fputs("+\n",fp);
		if(NULL==seq->qual){
			warnx("Attempted to display qualities of sequence with none.");
			for ( int i=0 ; i<seq->len ; i++){
				fputc(QUALITY_PHREDC_MISSING,fp);
			}
			if(has_ecc(seq)){
				fputc('\n',fp);
				for ( int i=0 ; i<seq->len ; i++){
					fputc(QUALITY_PHREDC_MISSING,fp);
				}
			}
		} else {
			for ( int i=0 ; i<seq->len ; i++){
				fprint_qual(fp,seq->qual[i]);
			}
			if(has_ecc(seq)){
				fputc('\n',fp);
				for ( int i=0 ; i<seq->len ; i++){
					fprint_qual(fp,seq->equal[i]);
				}
			}
		}
		fputc('\n',fp);
	}
}


bool has_qualities(SEQ seq){
	return (NULL!=seq->qual);
}

bool has_ecc(SEQ seq){
	return (NULL!=seq->ebase);
}

char * readLine(FILE *fp){
    char * cstr;
    char * ln = NULL;
    size_t len=0;
    #ifdef  _GNU_SOURCE
    len = getline(&ln,&len,fp);
    #else
    ln = fgetln(fp,&len);
    #endif

    // Return NULL if nothing to read
    // ! Does getline allocate memory in this case?
    if(NULL==ln || -1==len){ return NULL;}
    
    // Create terminated string to return
    cstr = calloc(len+1,sizeof(char));
    memcpy(cstr,ln,len*sizeof(char));
    if(cstr[len-1]=='\n'){ cstr[len-1]='\0'; }
    #ifdef _GNU_SOURCE
    free(ln);
    #endif

    return cstr;
}


SEQ read_seq( FILE * fp){
        size_t len=0;
	if(NULL==fp){ return NULL; }

	// Read name
	char * line = readLine(fp);
	if(NULL==line){ return NULL; }
	len = strlen(line);
	if(line[0]!='>'){
		warnx("Expecting > character in file");
		return NULL;
	}
	char * name = calloc(len,sizeof(char));
	if(NULL==name){ return NULL; }
	strncpy(name,line+1,len);
        free(line);

	// Data line
	line = readLine(fp);
	if(NULL==line){ free(name); return NULL; }
	len = strlen(line);
	unsigned int seqlen;
	for( seqlen=0 ; line[seqlen]!='\t' && seqlen<len ; seqlen++); 
	seqlen--; // Don't count initBase


	SEQ seq = new_SEQ(seqlen,true,true);
	seq->name = name;

	seq->initBase = char2base(line[0]);
	int ptr=1;
	for ( int i=0 ; i<seqlen ; i++,ptr++){
		seq->base[i] = char2base(line[ptr]);
	}
	if(line[ptr++]!='\t'){ warnx("Parse error, expecting tab"); goto cleanup; }
	for ( int i=0 ; i<seqlen ; i++,ptr++){
		seq->qual[i] = phredc2qual(line[ptr]);
	}
	if(line[ptr++]!='\t'){ warnx("Parse error, expecting tab"); goto cleanup; }
	if(char2base(line[ptr++])!=seq->initBase){ warnx("Initial base of ecc reads does not match colors"); }
	for ( int i=0 ; i<seqlen ; i++,ptr++){
		// ECC bases given in probe space not template space, add T to convert back
		seq->ebase[i] = gf4_add(baseT,char2base(line[ptr]));
	}
	if(line[ptr++]!='\t'){ warnx("Parse error, expecting tab"); goto cleanup; }
	for ( int i=0 ; i<seqlen ; i++,ptr++){
		seq->equal[i] = phredc2qual(line[ptr]);
	}
        free(line);

	return seq;

cleanup:
	free_SEQ(seq);
	return NULL;
}


#ifdef TEST
char name[] = "427_19_1309_F3";
unsigned int len = 50;
char fastq_base[] = {
	'0','0','2','1','0','0','3','1','3','1',
	'2','0','0','1','3','0','3','1','3','0',
	'3','1','2','1','3','0','3','3','3','3',
	'0','3','1','2','2','1','1','3','2','2',
	'1','0','1','0','2','0','0','0','0','2'};
phredq pqual[] = {22,30,19,9,21,27,28,16,25,21,24,25,20,20,24,26,26,23,11,25,23,16,18,4,19,28,20,24,7,28,27,22,25,4,20,12,9,29,4,24,20,19,20,4,25,21,25,25,4,23};

int main (int argc, char * argv[]){
	SEQ seq = new_SEQ(len,true,false);
	
	seq->name = calloc(1+strlen(name),sizeof(char));
	strcpy(seq->name,name);
	for ( int i=0 ; i<len ; i++){
		seq->base[i] = char2base(fastq_base[i]);
	}
	for ( int i=0 ; i<len ; i++){
		seq->qual[i] = phredq2qual(pqual[i]);
	}
	fprint_SEQ(stdout,seq,false,true);
	fprint_SEQ(stdout,seq,true,false);
	fprint_SEQ(stdout,seq,true,true);
	free_SEQ(seq);
}
#endif /* TEST */
