#include <stdio.h>
#include <stdlib.h>
#include "gf4.h"

/** Add two elements of GF(4) together
 * Addition in GF4 is bitwise exclusive or
 */
base gf4_add( const base b1, const base b2){
	if(baseN==b1 || baseN==b2){ return baseN;}
	return b1 ^ b2;
}

/** Multiply two elements of GF(4) together
 */
base gf4_mult(const base b1, const base b2){
	if(baseA==b1 || baseA==b2){ return baseA;} // baseA is zero element
	if(baseN==b1 || baseN==b2){ return baseN;}
	if(baseC==b1 || baseC==b2){ return baseC ^ b1 ^ b2;} // baseC is identity element
	// b1 and b2 are either baseG or baseT
	// If b1!=b2, then b1.b2=C
	if(b1!=b2){ return baseC;}
	return (baseG==b1)?baseT:baseG;
}

base * gf4_vector_matrix(const base * bseq, const base * mat, const int nrow, const int ncol, base * res){
	if(NULL==bseq){ return NULL;}
	if(NULL==mat){ return NULL;}
	if(NULL==res){
		res = calloc(ncol,sizeof(base));
		if(NULL==res){ return NULL;}
	}

	for ( int i=0 ; i<ncol ; i++){
		res[i] = baseA;
		for( int j=0 ; j<nrow ; j++){
			res[i] = gf4_add( res[i], gf4_mult(bseq[j],mat[i*nrow+j]) );
		}
	}

	return res;
}

void fprint_table(FILE * fp, const char * s, base (*fun)(base,base) ){
	if(NULL==fp){ return;}
	if(NULL==fun){ return;}

	if(NULL!=s){ fputs(s,fp);}
	fputs("\n |ACGT\n-+----\n",fp);
	for ( base i=0 ; i<4 ; i++){
		fprint_base(fp,i);
		fputc('|',fp);
		for ( base j=0 ; j<4 ; j++){
			fprint_base(fp,fun(i,j));
		}
		fputc('\n',fp);
	}
}


#ifdef TEST

static base testBlock1[] = { baseC, baseG, baseA, baseG, baseA};
static base testBlock2[] = { baseA, baseC, baseA, baseA, baseT};
static base resBlock1[] =  { baseT, baseG, baseG, baseG, baseG};
static base resBlock2[] =  { baseC, baseC, baseA, baseT, baseT};

int main(int argc, char * argv[]){
	base res[BLOCK_SIZE] = {0};

	// Cayley tables
	fprint_table(stdout,"gf4_add",gf4_add);
	fputc('\n',stdout);
	fprint_table(stdout,"gf4_mult",gf4_mult);
	fputc('\n',stdout);

	// First test block. Answer is TGGGC
	fputs("Encoding of ",stdout);
	fprint_block(stdout,testBlock1);
	fputs(" = ",stdout);
	gf4_vector_matrix(testBlock1,encodeMat,BLOCK_SIZE,BLOCK_SIZE,res);
	fprint_block(stdout,res);
	fputs(" (expecting ",stdout);
	fprint_block(stdout,resBlock1);
	fputs(")\n",stdout);

	// Second test block. Answer is CCATT
	fputs("Encoding of ",stdout);
	fprint_block(stdout,testBlock2);
	fputs(" = ",stdout);
	gf4_vector_matrix(testBlock2,encodeMat,BLOCK_SIZE,BLOCK_SIZE,res);
	fprint_block(stdout,res);
	fputs(" (expecting ",stdout);
	fprint_block(stdout,resBlock2);
	fputs(")\n",stdout);

	// Third test block. Answer is CGAGA
	fputs("Decoding of ",stdout);
	fprint_block(stdout,resBlock1);
	fputs(" = ",stdout);
	gf4_vector_matrix(resBlock1,decodeMat,BLOCK_SIZE,BLOCK_SIZE,res);
	fprint_block(stdout,res);
	fputs(" (expecting ",stdout);
	fprint_block(stdout,testBlock1);
	fputs(")\n",stdout);

	// Fourth test block. Answer is ACAAT
	fputs("Decoding of ",stdout);
	fprint_block(stdout,resBlock2);
	fputs(" = ",stdout);
	gf4_vector_matrix(resBlock2,decodeMat,BLOCK_SIZE,BLOCK_SIZE,res);
	fprint_block(stdout,res);
	fputs(" (expecting ",stdout);
	fprint_block(stdout,testBlock2);
	fputs(")\n",stdout);


	return EXIT_SUCCESS;
}
#endif /* TEST */

