#include <stdlib.h>
#include <getopt.h>
#include <err.h>
#include <string.h>
#include <assert.h>
#include <stdbool.h>
#include <time.h>
#include "SFMT-src-1.3/SFMT.h"
#include "conversion.h"
#include "sequence.h"
#include "mutate_color.h"

#define PROGNAME "simSolid"

static struct option longopts[] = {
	{"generator", required_argument, NULL, 'g'},
	{"initial", required_argument, NULL, 'i'},
	{"mutate",  no_argument, NULL, 'm'},
	{"noecc", no_argument, NULL, 'n'},
	{"seed", required_argument, NULL, 's'},
	{"help", no_argument, NULL, 'h'},
	{"licence", no_argument, NULL, 0},
	{"license", no_argument, NULL, 0},
	{NULL,0,NULL,0}
};


typedef struct options {
	bool ecc,mutate;
	base initBase;
	base probeGen[MAXCODE];
	uint32_t seed;
} OPT;

OPT opt = {true,true,baseT,{baseC,baseT,baseA,baseT},0};

void fprint_usage(FILE * fp){
	fputs(
"\t" PROGNAME "\n"
"Simulate SOLiD4 color data\n"
"\n"
"Usage:\n"
"\t" PROGNAME " [-g generator] [-i initBase] -mn [-s seed] file\n"
"\t" PROGNAME " --help\n"
"\t" PROGNAME " --licence\n"
"\n"
PROGNAME " reads from a fasta formated file containing sequence"
"and prints a tim formated file containing colors and ecc to stdout.\n"
"\n"
"Example: " PROGNAME " -n reads.fa\n"
	,fp);
}

void fprint_help(FILE * fp){
	fputs(
"\n"
"\n"
"-g, --generator probeGen [ default: CTAT]\n"
"\tProbe generator for code of second probe set.\n"
"\n"
"-i, --initial initialBase [default: T]\n"
"\tInitial base for direct translation of color-space information\n"
"\n"
"-m, --mutate [default: true]\n"
"\tToggle mutation\n"
"\n"
"-n, --noecc [default: false]\n"
"\tDon't simulate ECC colors\n"
"\n"
"-s, --seed [default: from clock]\n"
"\tSeed for random number generator. 0==from clock\n"
"\n"
"-h, --help\n"
"\tDisplay information about usage and exit.\n"
"\n"
"--licence, --license\n"
"\tDisplay licence terms and exit.\n"
, fp);
}


void fprint_licence (FILE * fp){
	fputs(
		"  " PROGNAME " : Simulate SOLiD4 color data\n"
		#include "copyright.inc"
	,fp);
}



void parse_options(int argc, char * argv[]){
	int ch;
	while ((ch = getopt_long(argc, argv, "g:i:mnh", longopts, NULL)) != -1){
		switch(ch){
			case 'g': if(strlen(optarg)>MAXCODE){
                                          warnx("Generator for code is too long, will truncate to %d positions",MAXCODE);
                                  } else if (strlen(optarg)<MAXCODE){
                                          warnx("Generator for code is too short, will pad with zeros");
                                  }
                                  int i = 0;
                                  for ( i=0 ; i<strlen(optarg) ; i++){
                                          base b = char2base(optarg[i]);
                                          if(b==baseN){
                                                  errx(EXIT_FAILURE,"Generator %s contains illegal base %c",optarg,optarg[i]);
                                          }
                                          opt.probeGen[i] = b;
                                  }
                                  for ( ; i<MAXCODE ; i++){
                                          opt.probeGen[i] = baseA;
                                  }
				  break;
			case 'i': opt.initBase = char2base(optarg[0]);
				  break;
			case 'm': opt.mutate = !opt.mutate;
			      break;
			case 'n': opt.ecc = !opt.ecc;
				  break;
			case 's': opt.seed = strtoul(optarg,NULL,10);
				  break;
			case 'h': fprint_usage(stdout);
				  fprint_help(stdout);
				  exit(EXIT_SUCCESS);
			case 0:	  fprint_licence(stdout);
				  exit(EXIT_SUCCESS);
			default:  warnx("Unrecognised option %s",argv[optind]);
				  fprint_usage(stderr);
				  exit(EXIT_FAILURE);
		}
	}
}


int main ( int argc, char * argv[] ){
	parse_options(argc,argv);
	argc -= optind;
	argv += optind;
	
	if(opt.seed==0){
		opt.seed = (uint32_t) time(NULL);
		fprintf(stderr,"Using seed %u\n",opt.seed);
	}
	init_gen_rand( opt.seed );

	FILE * fp = stdin;
	size_t ctr=0;
	do {
		// Open next file, or use stdin
		if(argc>0){
			fp = fopen(argv[0],"r");
			if(NULL==fp){
				warnx("Failed to open %s for input",argv[0]);
			} else {
				fprintf(stderr,"Reading from %s\n",argv[0]);
			}
		} else {
			fputs("Reading from stdin\n",stderr);
		}

		/* Read file and convert sequence */
		SEQ origseq = NULL;
		while( (origseq = read_fasta(fp))!=NULL ){
			assert(!has_qualities(origseq));
			assert(!has_ecc(origseq));
			// Convert sequence into ECC space
			SEQ seq = copy_SEQ(origseq);
			if(opt.ecc){
				seq->ebase = convert_sequence_to_ecc(seq->base,seq->len,opt.probeGen);
			}
			convert_sequence_to_color_inplace(seq->base,opt.initBase,seq->len);
			seq->initBase = opt.initBase;
			// Simulate qualities
			phredq initPhred = generate_initial_phredq();
			seq->qual = generate_qualities(initPhred,seq->len);
			if(opt.ecc){
				seq->equal = generate_qualities(initPhred,seq->len);
			}
			// Mutate sequence
			int nmut = 0;
			if(opt.mutate){
				nmut = mutate_sequence_inplace(seq->base,seq->qual,seq->len,0.);
				if(opt.ecc){
					mutate_sequence_inplace(seq->ebase,seq->equal,seq->len,0.);
				}
			}
			if(true/*nmut<=5*/){
				// Print Tim format file
				fprint_timSEQ(stdout,seq,origseq);
			}

			free_SEQ(seq); seq = NULL;
			free_SEQ(origseq); origseq = NULL;
			ctr++;
			if((ctr%10000)==0){
			   fprintf(stderr,"\rDone %8u",ctr);
			}
		}
		fputc('\n',stderr);

		// Close file and set up for next
		fclose(fp);
		argc--;
		argv++;
	} while(argc>0);
}

