#include <math.h>
#include <assert.h>
#include <err.h>
#include <stdlib.h>
#include <stdbool.h>
#include "SFMT-src-1.3/SFMT.h"

double powm1(const double x, const double y){
	return expm1(y*log(x));
}

double rstdunif(void){
	return genrand_real3();
}

/** Choose a random element from a vector of probabilities.
 */
int choose_elt(const double * x, const int n){
   double p = genrand_real3();
   int i=0;
   while(i<n && p>x[i]){ p-=x[i]; i++; }
   if(i>=n){ errx(EXIT_FAILURE,"Element out of range in choose_elt"); }

   return i;
}

/** Random gamma variable when shape>1
  * See Knuth V2, 3.41.
  */
double rgamma1( const double shape){
	assert(shape>1);
	double v,x,y;
	const double shm1 = shape - 1.0;
	const double sh2m1 = sqrt(2.0*shape-1.0);

start:
	y = tan(M_PI*genrand_real3());
	x = sh2m1*y + shm1;

	if(x<=0.0){ goto start;}
	
	v = genrand_real3();
	if(v>(1.0+y*y)*exp(shm1*log(x/shm1)-sh2m1*y)){ goto start;}
	return x;
}

/** Random gamma variable when shape<1
  * See Kundu and Gupta 2007
  * "A convenient way of generating gamma random variables using generalized exponential distribution"
  */
double rgamma2 ( const double shape){
	assert(shape>0 && shape<1);
	double u,v,x;
	const double d = 1.0334 - 0.0766*exp(2.2942*shape);
	const double a = exp2(shape) * pow(-expm1(-d/2),shape);
	const double pdsh = pow(d,shape-1.0);
	const double b = shape * pdsh * exp(-d);
	const double c = a+b;
	
start:
	u = genrand_real3();
	x = (u<=a/c) ? -2.0 * log1p(-pow(c*u,1.0/shape)/2.0) : -log( c*(1.0-u)/(shape*pdsh) );
	v = genrand_real3();
	if(x<=d){
		const double p = pow(x,shape-1.0)*exp(-x/2.0)/( exp2(shape-1.0)*pow(-expm1(-x/2.0),shape-1.0) );
		if(v>p){ goto start; }
	} else {
		if(v>pow(d/x,1.0-shape)){ goto start;}
	}
	
	return x;
}

/* Exponential distribution with rate */
double rexp(const double r){
	return -log(genrand_real3())/r;
}


double rgamma(const double shape, const double scale){
	assert(shape>0.0 && scale>0.0);
	if(shape==1.0){ return scale*rexp(1.0); }
	if(shape>1.0){ return scale*rgamma1(shape); }
	return scale*rgamma2(shape);
}


double rchisq(const double df){
	assert(df>0.0);
	return rgamma(df/2.0,2.0);
}


// Random normal by Box Muller
double rstdnorm(){
   static bool empty=true;
   static double y = NAN;
   if(!empty){
      empty = true;
      return y;
   }

   double R = sqrt(-2.0 * log(genrand_real3()) );
   double theta = 2.0 * M_PI * genrand_real3();

   y = R * cos(theta);
   empty = false;
   return R*sin(theta);
}


double rnorm(const double mean, const double sd){
	assert(sd>0.0);
	return rstdnorm()*sd+mean;
}


double rskewnorm(const double location, const double scale, const double shape){
	assert(scale>0.0);
	double u1 = rstdnorm();
	double u2 = rstdnorm();
	if(u2>shape*u1){ u1 = -u1; }
	return location + scale*u1;
}

double rskewt(const double location, const double scale, const double shape, const double df){
	assert(scale>0.0);
	assert(df>0.0);
	double z = rskewnorm(0.0,scale,shape);
	double v = rchisq(df)/df;
	return z/sqrt(v) + location;
}

double rweibull( const double shape, const double scale){
	double e = rexp(1.0);
	return scale * pow(e,1.0/shape);
}

double rbeta ( const double a, const double b){
	double x = rgamma(a,1);
	double y = rgamma(b,1);
	return x/(x+y);
}

double rbetaprime( const double a, const double b){
	double x = rbeta(a,b);
	return x/(1.-x);
}

double rkumaraswamy ( const double a, const double b){
	double u = rstdunif();
	return pow(-powm1(u,1.0/b),1.0/a);
}

double rlognorm ( const double m, const double sd){
	return exp(rnorm(m,sd));
}

double rt( const double df){
	double z = rstdnorm();
	double v = rchisq(df);
	return z/sqrt(v/df);
}

double rcauchy( const double loc, const double scale){
	return rt(1.0)*scale + loc;
}

double rlevy ( const double loc, const double scale){
	double z = rnorm(0.0,1./sqrt(scale));
	return 1.0/(z*z);
}

double rinvgaussian( const double mu, const double lambda){
	double y = rchisq(1.0);
	double muy = mu*y;
	double lam2 = 2.0*lambda;
	double x = mu + mu*muy/lam2 - mu/lam2 * sqrt( 2.0*lam2*muy + muy*muy);
	double z = rstdunif();
	return ( rstdunif()<=mu/(mu+x) ) ? x : mu*mu/x;
}


#ifdef TEST
#include <stdint.h>
#include <time.h>
int main(int argc, char * argv[]){
    if(argc!=7){
    	fputs("rgen seed loc scale shape df n\n",stderr);
    	exit(EXIT_FAILURE);
    }
    uint32_t seed = strtoul(argv[1],NULL,0);
    if(seed==0){ seed = (uint32_t) time(NULL); }
    fprintf(stderr,"Using seed %u\n",seed);

    init_gen_rand( seed );
    double loc = strtod(argv[2],NULL);
    double scale = strtod(argv[3],NULL);
	double shape = strtod(argv[4],NULL);
	double df = strtod(argv[5],NULL);
	unsigned int n = strtoul(argv[6],NULL,10);
	for ( int i=0 ; i<n ; i++){
		fprintf(stdout,"%e\n",rskewt(loc,scale,shape,df));
	}
	return EXIT_SUCCESS;
}
#endif /* TEST */
