#include <stdlib.h>
#include <math.h>
#include <stdbool.h>
#include <err.h>
#include "mutate_color.h"
#include "SFMT-src-1.3/SFMT.h"
#include "rgenerate.h"

/* Mutational data from mapping syn-bead read in color space.
 * Single color error reads only.
 */
/*double mutations[4][4] = {
   { 0.0, 0.3533695683499976, 0.37166585824185433, 0.27496457340814806 },
   { 0.3680965003568155, 0.0, 0.39807529268515623, 0.23382820695802828 },
   { 0.36714652567431777, 0.377834763786192, 0.0, 0.2550187105394902},
   { 0.5147307218556777, 0.271384623827701, 0.21388465431662135, 0.0}
};*/
double mutations[4][4] = {
	{ 0.0, 0.333333, 0.333334, 0.333333},
	{ 0.333333, 0.0, 0.333333, 0.333334},
	{ 0.333334, 0.333333, 0.0, 0.333333},
	{ 0.333333, 0.333334, 0.333333, 0.0}
};


base mutate_base ( const base b, const prob p){
   if(baseN==b){ return baseN; }
   return (genrand_real3()>p)?b:choose_elt(mutations[b],4);
}

int mutate_sequence_withQualities_inplace ( base * b, const qual * qual, const int len){
   if(NULL==b){ return -1;}
   if(NULL==qual){ return -1;}

	int mutations = 0;
   for ( int i=0 ; i<len ; i++){
      base n = mutate_base(b[i],qual2_1mprob(qual[i]));
      if( b[i]!=n){ mutations++;}
      b[i] = n;
   }
   return mutations;
}

int mutate_sequence_withProb_inplace(base * b, const int len, const prob p){
   if(NULL==b){ return -1;}
   
    int mutations = 0;
   for ( int i=0 ; i<len ; i++){
      base n = mutate_base(b[i],p);
      if( b[i]!=n){ mutations++;}
      b[i] = n;
   }
   return mutations;
}


/** Mutate a sequence and ecc information if present.
 *  If qualities given then these as used for probability 
 *  of mutation, otherwise value of p is used.
 */
int mutate_sequence_inplace( base * b, const qual * qual, const int len, const double p){
   return (NULL==qual)?mutate_sequence_withProb_inplace(b,len,p):mutate_sequence_withQualities_inplace(b,qual,len);
}




/* Fitted model to synbead data
 */
qual * generate_qualities(const phredq initPhred, const int len){
   qual * q = calloc(len,sizeof(qual));
   if(NULL==q){ return NULL; }

   // Variance is linearly related to initPhred
   
   // Slope from lm(formula = mean_qual ~ cy)
   const double slope1 = -0.326107;
   const double slope2 = 0.0;
   for ( int i=0 ; i<len ; i++){
      int cy = i/5;
      const double sd = exp( 0.5 * ( -0.5376705 + 0.40646631*initPhred - 0.01083223*initPhred*initPhred + 0.1169215 * cy ) );
      double phred = rstdnorm()*sd + initPhred;
      phred = phred+slope1*cy+slope2*log1p(cy);
      if(phred<1.249387){ phred = 1.249387; }
      q[i] = phredq2qual(phred);
      if(q[i]<-1.386294){ q[i] = -1.386294;} // Ensure P(correct)>=0.25
      // Generalised error
      q[i] += log(0.9999);
   }
   return q;
}

phredq generate_initial_phredq(void){
   // Generate random intercept
   // Parameters for skew-t, fitted to syn-bead data intercepts
   // Fitted skew-normal to mean for each cycle corrected for cycle
   // See: http://azzalini.stat.unipd.it/SN/faq-r.html
   const double location = 31.499732;
   const double scale = 5.133327;
   const double shape = -8.665066;
   const double df = 3.348463;
   
   return rskewt(location,scale,shape,df);
}


#ifdef TEST
#include <inttypes.h>
#include <stdint.h>
#include <time.h>


base b[] = { baseC,baseT,baseG,baseA,baseC,baseG,baseC,baseT,baseC,baseT,baseT,baseG,baseC,baseG,baseG,baseA,baseA,baseG,baseG,baseA};
qual q[] = {
	-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,
	-2.3025850929940455,-0.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,
	-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,
	-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455,-2.3025850929940455};
int len = 20;

int main(int argc, char * argv[]){
   // Initialise random number generator
   uint32_t seed = (argc<2)? (uint32_t) time(NULL) : strtoul(argv[1],NULL,0);
   fprintf(stderr,"Using seed %u\n",seed);
   init_gen_rand( seed );

   fprint_baseVec(stdout,b,len);
   fputs(" => ",stdout);
   mutate_sequence_inplace(b,q,len,0.0);
   fprint_baseVec(stdout,b,len);
   fputc('\n',stdout);


   phredq pq = generate_initial_phredq();
   qual * q = generate_qualities(pq,len);
   for( int i=0 ; i<len ; i++){
      fprint_qual(stdout,q[i]);
   }
   fputc('\n',stdout);
   fprint_baseVec(stdout,b,len);
   fputs(" => ",stdout);
   mutate_sequence_inplace(b,q,len,0.0);
   fprint_baseVec(stdout,b,len);
   fputc('\n',stdout);
}
#endif /* TEST */

